/* global CONFIG */
(function(){
  const $ = (s)=>document.querySelector(s);
  const audio   = $('#audio');
  const playBtn = $('#playBtn');
  const muteBtn = $('#muteBtn');
  const vol = $('#volume');
  const cover = $('#cover');
  const trackEl = $('#track');
  const artistEl = $('#artist');
  const openPopup = document.getElementById('openPopup');

  // -------- Web Audio chain (admin-controlled; no UI toggle) --------
  let ctx, source, gain, comp, lowShelf, midPeak, highShelf;
  function ensureAudioGraph(){
    if(ctx) return;
    ctx = new (window.AudioContext || window.webkitAudioContext)();
    source = ctx.createMediaElementSource(audio);

    if(CONFIG.enhance){
      gain = ctx.createGain(); gain.gain.value = 1.18;
      comp = ctx.createDynamicsCompressor();
      comp.threshold.value = -18; comp.knee.value = 18; comp.ratio.value = 2.2; comp.attack.value = 0.005; comp.release.value = 0.15;
      lowShelf = ctx.createBiquadFilter(); lowShelf.type = 'lowshelf'; lowShelf.frequency.value = 95; lowShelf.gain.value = 2.5;
      midPeak = ctx.createBiquadFilter(); midPeak.type = 'peaking'; midPeak.frequency.value = 3000; midPeak.Q.value = 1; midPeak.gain.value = -1.0;
      highShelf = ctx.createBiquadFilter(); highShelf.type = 'highshelf'; highShelf.frequency.value = 8500; highShelf.gain.value = 2.0;
      source.connect(lowShelf); lowShelf.connect(midPeak); midPeak.connect(highShelf); highShelf.connect(comp); comp.connect(gain); gain.connect(ctx.destination);
    } else {
      source.connect(ctx.destination);
    }
  }

  // -------- Controls --------
  playBtn.addEventListener('click', async ()=>{
    ensureAudioGraph();
    if(audio.paused){
      try { await audio.play(); playBtn.textContent = '⏸'; } catch(e){ console.error(e); }
    } else {
      audio.pause(); playBtn.textContent = '▶';
    }
  });
  muteBtn.addEventListener('click', ()=>{
    audio.muted = !audio.muted;
    muteBtn.textContent = audio.muted ? '🔈' : '🔇';
  });
  vol.addEventListener('input', ()=> audio.volume = parseFloat(vol.value));
  if(openPopup){ openPopup.addEventListener('click', ()=>{ const url = new URL(window.location.href); window.open(url,'_blank','noopener,width=480,height=760'); }); }

  // -------- Metadata + cover --------
  let lastTitle = '';
  async function fetchJSON(u){
    const res = await fetch(u, {cache:'no-store'});
    if(!res.ok) throw new Error('HTTP '+res.status);
    return res.json();
  }
  function updateUI(meta){
    const artist = meta.artist || '';
    const titleCandidate = meta.title || meta.streamTitle || '';
    const title = titleCandidate || CONFIG.notitle || 'En vivo';
    trackEl.textContent = title;
    artistEl.textContent = artist || '';
    const fullTitle = [CONFIG.name, title].filter(Boolean).join(' | ');
    document.title = fullTitle;

    if((titleCandidate && titleCandidate !== lastTitle) || (!titleCandidate && lastTitle === '')){
      lastTitle = titleCandidate || '';
      const q = new URLSearchParams({ artist: artist, title: titleCandidate, fallback: CONFIG.fallback, logo: CONFIG.logo });
      fetch('api/cover.php?'+q.toString()).then(r=>r.json()).then(d=>{ if(d && d.image){ cover.src = d.image; } }).catch(()=>{});
    }
  }
  async function pollMetadata(){
    const q = new URLSearchParams({ stream: CONFIG.stream, type: CONFIG.type, mount: CONFIG.mount });
    try{ const meta = await fetchJSON('api/metadata.php?'+q.toString()); updateUI(meta); }
    catch(e){ console.warn('metadata error', e); }
    finally{ setTimeout(pollMetadata, 10000); }
  }

  // Initialize: full volume and try autoplay
  try { audio.volume = 1.0; vol && (vol.value = 1.0); audio.muted = false; } catch(e) {}
  (async ()=>{
    try {
      ensureAudioGraph();
      await audio.play();
      playBtn.textContent = '⏸';
    } catch(e) { console.warn('autoplay blocked by browser policy'); }
  })();

  pollMetadata();
})();
