<?php
declare(strict_types=1);
header('X-Frame-Options: SAMEORIGIN');
?>
<!doctype html>
<html lang="es">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Generador de Reproductor de Radio</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;800&display=swap" rel="stylesheet">
<link rel="stylesheet" href="assets/styles.css">
<style>
body{max-width:980px;margin:0 auto;padding:24px}
.form{background:#11161b;border:1px solid #26313b;border-radius:16px;padding:20px}
.form h1{margin:0 0 12px}
.grid{display:grid;gap:12px;grid-template-columns:1fr 1fr}
.grid .col-2{grid-column:span 2}
label{font-weight:600;margin-bottom:6px;display:block}
input,select,button,textarea{width:100%;padding:10px;border-radius:10px;border:1px solid #2b3946;background:#0f141a;color:#e6f0ff}
button{cursor:pointer;background:#1f6feb;border:1px solid #2b6de6;font-weight:700}
button.secondary{background:#19232b;border:1px solid #2b3946}
.output{margin-top:18px}
.code{background:#0b0f13;border:1px dashed #2b3946;border-radius:12px;padding:12px;font-family:ui-monospace,Consolas,monospace;overflow:auto}
small{opacity:.8}
.preview{margin-top:18px;border:1px dashed #2b3946;border-radius:12px;overflow:hidden}
.note{margin-top:6px;color:#9bb5cc;font-size:12px}
</style>
</head>
<body>
  <div class="form">
    <h1>Generar reproductor (iframe)</h1>
    <p>Sistema multi-player. Elige diseño, tamaño y configura el mensaje por defecto cuando no haya título. La mejora de audio se activa desde aquí (no hay botón en el player).</p>
    <form id="gen">
      <div class="grid">
        <div>
          <label for="name">Nombre de la emisora</label>
          <input id="name" name="name" placeholder="AZ Radio Latina" required>
        </div>
        <div>
          <label for="stream">URL del stream (MP3/AAC)</label>
          <input id="stream" name="stream" placeholder="https://s2.pcpcomunicaciones.com:8002/stream" required>
        </div>
        <div>
          <label for="type">Tipo de servidor</label>
          <select id="type" name="type">
            <option value="">Auto (detectar)</option>
            <option value="icecast">Icecast</option>
            <option value="shoutcast">Shoutcast</option>
          </select>
        </div>
        <div>
          <label for="mount">Mount / SID (opcional)</label>
          <input id="mount" name="mount" placeholder="/stream ó sid=1">
        </div>
        <div>
          <label for="logo">Logo de la emisora (URL)</label>
          <input id="logo" name="logo" value="https://upload.wikimedia.org/wikipedia/commons/a/ab/Logo_TV_2022.svg">
          <div class="note">Si no hay portada, se usará el <strong>logo</strong> como cover.</div>
        </div>
        <div>
          <label for="layout">Layout</label>
          <select id="layout" name="layout">
            <option value="square">Cuadrado</option>
            <option value="vertical">Vertical</option>
            <option value="horizontal">Horizontal</option>
            <option value="round">Redondo</option>
          </select>
        </div>
        <div>
          <label for="size">Tamaño</label>
          <select id="size" name="size">
            <option value="sm">Pequeño</option>
            <option value="md" selected>Mediano</option>
            <option value="lg">Grande</option>
            <option value="lx">Extra grande (LX)</option>
          </select>
        </div>
        <div class="col-2">
          <label for="fallback">Imagen de respaldo (URL) para la portada</label>
          <input id="fallback" name="fallback" value="https://via.placeholder.com/600x600.png?text=Radio">
        </div>
        <div class="col-2">
          <label for="notitle">Mensaje si no hay título (placeholder)</label>
          <input id="notitle" name="notitle" value="¡Estamos al aire!">
          <div class="note">Se mostrará cuando el servidor no devuelva título/metadata.</div>
        </div>
        <div class="col-2">
          <label><input type="checkbox" id="enhance" name="enhance" checked> Activar mejora de audio (compresor/EQ) desde el admin</label>
        </div>
        <div class="col-2">
          <label for="accent">Color de acento (hex)</label>
          <input id="accent" name="accent" value="#ff2d55">
        </div>
      </div>
      <div style="display:flex;gap:10px;margin-top:14px">
        <button type="submit">Generar iframe</button>
        <button type="button" class="secondary" id="previewBtn">Previsualizar</button>
        <button type="button" class="secondary" id="previewAllBtn">Previsualizar todos</button>
      </div>
    </form>
    <div class="output" id="out" hidden>
      <h3>HTML para embeber</h3>
      <div class="code"><code id="embed"></code></div>
      <p><small>La altura se ajusta según layout/tamaño. El player no tiene botón de mejora; abrir ventana y compartir son iconos pequeños en la esquina.</small></p>
      <div class="preview" id="previewBox" hidden>
        <iframe id="previewFrame" src="" width="100%" height="560" frameborder="0" allow="autoplay" loading="lazy"></iframe>
      </div>
      <div class="preview" id="gridBox" hidden>
        <div id="grid" style="display:grid;gap:14px;grid-template-columns:repeat(auto-fit,minmax(260px,1fr));padding:12px"></div>
      </div>
    </div>
  </div>

<script>
const form = document.getElementById('gen');
const out = document.getElementById('out');
const code = document.getElementById('embed');
const prevBtn = document.getElementById('previewBtn');
const previewBox = document.getElementById('previewBox');
const previewFrame = document.getElementById('previewFrame');
const previewAllBtn = document.getElementById('previewAllBtn');

function buildURL(params){
  const q = new URLSearchParams(params);
  if(!q.get('enhance')) q.set('enhance','0'); else q.set('enhance','1');
  return 'player.php?' + q.toString();
}
function computeHeight(layout, size){
  const map = {
    square:   { sm: 520, md: 560, lg: 640, lx: 800 },
    vertical: { sm: 640, md: 760, lg: 900, lx: 1280 },
    horizontal:{sm: 360, md: 420, lg: 480, lx: 620 },
    round:    { sm: 520, md: 560, lg: 640, lx: 800 },
  };
  return (map[layout] && map[layout][size]) ? map[layout][size] : 560;
}
function buildIframe(url, layout, size){
  const h = computeHeight(layout, size);
  return `<iframe src="${url}" width="100%" height="${h}" frameborder="0" allow="autoplay" loading="lazy"></iframe>`;
}
function getData(){ return Object.fromEntries(new FormData(form).entries()); }

function render(){
  const data = getData();
  const url = buildURL(data);
  out.hidden = false; previewBox.hidden = false;
  code.textContent = buildIframe(url, data.layout, data.size);
  previewFrame.src = url;
  previewFrame.height = computeHeight(data.layout, data.size);
}
function buildAll(){
  const data = getData();
  const layouts = ['square','horizontal','vertical','round'];
  const sizes = ['sm','md','lg','lx'];
  const grid = document.getElementById('grid');
  const gridBox = document.getElementById('gridBox');
  grid.innerHTML='';
  layouts.forEach(layout=>{
    sizes.forEach(size=>{
      const params = {...data, layout, size};
      const url = buildURL(params);
      const h = computeHeight(layout, size);
      const card = document.createElement('div');
      card.style.background = '#0b0f13';
      card.style.border = '1px solid #26313b';
      card.style.borderRadius = '12px';
      card.style.padding = '8px';
      card.innerHTML = `<div style="display:flex;align-items:center;justify-content:space-between;margin:4px 6px 8px 6px">
        <strong>${layout} / ${size}</strong>
        <code style="opacity:.7;font-size:12px">${h}px</code>
      </div>
      <iframe src="${url}" width="100%" height="${h}" frameborder="0" allow="autoplay" loading="lazy"></iframe>`;
      grid.appendChild(card);
    });
  });
  out.hidden = false;
  gridBox.hidden = false;
  window.scrollTo({top:document.body.scrollHeight,behavior:'smooth'});
}

form.addEventListener('submit', (e)=>{ e.preventDefault(); render(); });
prevBtn.addEventListener('click', render);
previewAllBtn.addEventListener('click', buildAll);
</script>
</body>
</html>
